// Add polyfill for crypto.randomUUID
if (typeof crypto !== 'undefined' && !crypto.randomUUID) {
    crypto.randomUUID = function() {
        return ([1e7]+-1e3+-4e3+-8e3+-1e11).replace(/[018]/g, c =>
            (c ^ crypto.getRandomValues(new Uint8Array(1))[0] & 15 >> c / 4).toString(16)
        );
    };
}

class VideoPlayer {
    constructor() {
        this.art = null;
        this.hls = null;
        this.loadingSpinner = document.querySelector('.loading-spinner');
        this.errorMessage = document.querySelector('.error-message');
        this.lastUpdateTime = 0;
        this.isLocked = false;
        this.isMobile = /iPhone|iPad|iPod|Android/i.test(navigator.userAgent);
        this.controlsTimeout = null;
        this.isControlsVisible = false;
    }

    // Initialize the player
    init(videoUrl) {
        // Validate URL
        if (!this.isValidUrl(videoUrl)) {
            this.showError('Invalid video URL provided');
            return;
        }

        // Check if required libraries are loaded
        if (!window.Artplayer || !window.Hls) {
            this.showError('Required libraries not loaded');
            return;
        }

        try {
            this.initializePlayer(videoUrl);
            this.attachEventListeners();
            if (this.isMobile) {
                this.setupMobileOptimizations();
            }
            this.setupControlsVisibility();
        } catch (error) {
            console.error('Player initialization error:', error);
            this.showError('Failed to initialize player');
        }
    }

    // Initialize Artplayer instance
    initializePlayer(videoUrl) {
        const options = {
            ...playerConfig.defaultSettings,
            container: '.artplayer-app',
            url: videoUrl,
            customType: {
                m3u8: (video, url) => this.handleHLS(video, url),
            },
            controls: [
                {
                    name: 'rewind',
                    position: 'left',
                    html: '<button class="art-control-rewind"><svg viewBox="0 0 24 24"><path d="M11.99 5V1l-5 5 5 5V7c3.31 0 6 2.69 6 6s-2.69 6-6 6-6-2.69-6-6h-2c0 4.42 3.58 8 8 8s8-3.58 8-8-3.58-8-8-8z"/></svg></button>',
                    index: 10,
                },
                {
                    name: 'forward',
                    position: 'left',
                    html: '<button class="art-control-forward"><svg viewBox="0 0 24 24"><path d="M12 5V1l5 5-5 5V7c-3.31 0-6 2.69-6 6s2.69 6 6 6 6-2.69 6-6h2c0 4.42-3.58 8-8 8s-8-3.58-8-8 3.58-8 8-8z"/></svg></button>',
                    index: 11,
                }
            ],
            i18n: playerConfig.i18n
        };

        this.art = new Artplayer(options);

        // Add click handlers for rewind/forward buttons after player is initialized
        this.art.on('ready', () => {
            const rewindButton = this.art.template.$player.querySelector('.art-control-rewind');
            const forwardButton = this.art.template.$player.querySelector('.art-control-forward');

            if (rewindButton) {
                rewindButton.addEventListener('click', () => this.rewind());
            }

            if (forwardButton) {
                forwardButton.addEventListener('click', () => this.forward());
            }
        });

        // Add lock button for mobile
        if (this.isMobile) {
            this.addLockButton();
        }

        // Restore last playback position
        if (playerConfig.playbackMemory.enabled) {
            this.restorePlaybackPosition(videoUrl);
        }
    }

    // Setup mobile optimizations
    setupMobileOptimizations() {
        // Force volume to 100% on mobile
        if (this.art) {
            this.art.volume = 1;
            this.art.on('ready', () => {
                this.art.volume = 1;
            });
        }

        // Add double tap to seek
        let lastTapTime = 0;
        let lastTapX = 0;
        const doubleTapDelay = 300;

        this.art.on('touchstart', (event) => {
            const currentTime = new Date().getTime();
            const tapX = event.touches[0].clientX;
            const timeDiff = currentTime - lastTapTime;

            if (timeDiff < doubleTapDelay && Math.abs(tapX - lastTapX) < 30) {
                // Double tap detected
                if (tapX < window.innerWidth / 2) {
                    this.rewind();
                } else {
                    this.forward();
                }
                event.preventDefault();
            }

            lastTapTime = currentTime;
            lastTapX = tapX;
        });
    }

    // Add lock button
    addLockButton() {
        const lockButton = document.createElement('div');
        lockButton.className = 'art-lock-button';
        lockButton.innerHTML = `
            <svg viewBox="0 0 24 24">
                <path d="M18 8h-1V6c0-2.76-2.24-5-5-5S7 3.24 7 6v2H6c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h12c1.1 0 2-.9 2-2V10c0-1.1-.9-2-2-2zm0 12H6V10h12v10zm-6-3c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2z"/>
            </svg>
        `;

        this.art.container.appendChild(lockButton);

        lockButton.addEventListener('click', () => {
            this.toggleLock();
        });
    }

    // Setup controls visibility
    setupControlsVisibility() {
        if (!this.art) return;

        // Show controls initially
        this.showControls();

        // Handle mouse movement on desktop
        if (!this.isMobile) {
            this.art.template.$player.addEventListener('mousemove', () => {
                this.showControls();
            });

            this.art.template.$player.addEventListener('mouseleave', () => {
                if (!this.art.playing) return;
                this.hideControls();
            });
        } else {
            // Mobile-specific touch handling
            let touchStartTime = 0;
            let touchStartX = 0;
            let touchStartY = 0;

            this.art.template.$player.addEventListener('touchstart', (e) => {
                touchStartTime = Date.now();
                touchStartX = e.touches[0].clientX;
                touchStartY = e.touches[0].clientY;
            });

            this.art.template.$player.addEventListener('touchend', (e) => {
                const touchEndTime = Date.now();
                const touchDuration = touchEndTime - touchStartTime;
                
                // Only toggle controls for short taps (less than 200ms)
                if (touchDuration < 200) {
                    const touchEndX = e.changedTouches[0].clientX;
                    const touchEndY = e.changedTouches[0].clientY;
                    
                    // Calculate distance moved
                    const distance = Math.sqrt(
                        Math.pow(touchEndX - touchStartX, 2) + 
                        Math.pow(touchEndY - touchStartY, 2)
                    );
                    
                    // Only toggle if it's a tap (moved less than 10px)
                    if (distance < 10) {
                        if (this.isControlsVisible) {
                            // Only hide controls if video is playing
                            if (this.art.playing) {
                                this.hideControls();
                            }
                        } else {
                            this.showControls();
                        }
                    }
                }
            });
        }

        // Handle control interaction
        const controlElements = this.art.template.$player.querySelectorAll('.art-controls, .art-control-progress');
        controlElements.forEach(element => {
            element.addEventListener('touchstart', (e) => {
                e.stopPropagation();
                // Keep controls visible when interacting with them
                this.showControls();
                this.clearHideControlsTimer();
            });

            if (!this.isMobile) {
                element.addEventListener('mouseenter', () => {
                    this.clearHideControlsTimer();
                });
                element.addEventListener('mouseleave', () => {
                    if (this.art.playing) {
                        this.startHideControlsTimer();
                    }
                });
            }
        });

        // Show controls when video is paused
        this.art.on('pause', () => {
            this.showControls();
            this.clearHideControlsTimer();
        });

        // Handle controls visibility when playing
        this.art.on('play', () => {
            if (!this.isControlsVisible) {
                this.showControls();
            }
            this.startHideControlsTimer();
        });
    }

    // Show controls
    showControls() {
        if (!this.art) return;
        this.isControlsVisible = true;
        this.art.template.$player.classList.add('art-controls-show');
        // Always start the hide timer if video is playing
        if (this.art.playing) {
            this.startHideControlsTimer();
        }
    }

    // Hide controls
    hideControls() {
        if (!this.art || this.isLocked) return;
        this.isControlsVisible = false;
        this.art.template.$player.classList.remove('art-controls-show');
    }

    // Start timer to hide controls
    startHideControlsTimer() {
        this.clearHideControlsTimer();
        if (this.art.playing) {
            this.controlsTimeout = setTimeout(() => {
                this.hideControls();
            }, 3000); // Hide after 3 seconds
        }
    }

    // Clear hide controls timer
    clearHideControlsTimer() {
        if (this.controlsTimeout) {
            clearTimeout(this.controlsTimeout);
            this.controlsTimeout = null;
        }
    }

    // Toggle lock state
    toggleLock() {
        this.isLocked = !this.isLocked;
        if (this.isLocked) {
            this.art.container.classList.add('controls-locked');
            this.hideControls();
        } else {
            this.art.container.classList.remove('controls-locked');
            this.showControls();
        }

        // Update lock button icon
        const lockButton = this.art.container.querySelector('.art-lock-button');
        if (lockButton) {
            lockButton.innerHTML = this.isLocked ? `
                <svg viewBox="0 0 24 24">
                    <path d="M18 8h-1V6c0-2.76-2.24-5-5-5S7 3.24 7 6h2c0-1.66 1.34-3 3-3s3 1.34 3 3v2H6c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h12c1.1 0 2-.9 2-2V10c0-1.1-.9-2-2-2zm0 12H6V10h12v10zm-6-3c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2z"/>
                </svg>
            ` : `
                <svg viewBox="0 0 24 24">
                    <path d="M18 8h-1V6c0-2.76-2.24-5-5-5S7 3.24 7 6v2H6c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h12c1.1 0 2-.9 2-2V10c0-1.1-.9-2-2-2zM9 6c0-1.66 1.34-3 3-3s3 1.34 3 3v2H9V6zm9 14H6V10h12v10zm-6-3c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2z"/>
                </svg>
            `;
        }
    }

    // Handle HLS video type
    handleHLS(video, url) {
        if (Hls.isSupported()) {
            this.hls = new Hls();
            this.hls.loadSource(url);
            this.hls.attachMedia(video);
            this.hls.on(Hls.Events.ERROR, (event, data) => {
                this.handleHLSError(event, data);
            });
        } else if (video.canPlayType('application/vnd.apple.mpegurl')) {
            video.src = url;
        } else {
            this.showError('HLS is not supported in this browser');
        }
    }

    // Attach event listeners
    attachEventListeners() {
        if (!this.art) return;

        this.art.on('ready', () => {
            console.log('Player is ready');
            this.hideLoading();
        });

        this.art.on('error', () => {
            this.showError('Video playback error');
        });

        // Save playback position periodically
        if (playerConfig.playbackMemory.enabled) {
            this.art.on('timeupdate', () => {
                const now = Date.now();
                if (now - this.lastUpdateTime >= playerConfig.playbackMemory.updateInterval) {
                    this.savePlaybackPosition();
                    this.lastUpdateTime = now;
                }
            });
        }

        // Add keyboard controls
        document.addEventListener('keydown', (e) => this.handleKeyPress(e));
    }

    // Handle keyboard controls
    handleKeyPress(event) {
        if (!this.art) return;

        switch(event.key.toLowerCase()) {
            case ' ':
                this.art.toggle();
                break;
            case 'arrowleft':
                this.rewind();
                break;
            case 'arrowright':
                this.forward();
                break;
            case 'arrowup':
                if (!this.isMobile) {
                    this.art.volume = Math.min(1, this.art.volume + 0.1);
                }
                break;
            case 'arrowdown':
                if (!this.isMobile) {
                    this.art.volume = Math.max(0, this.art.volume - 0.1);
                }
                break;
            case 'f':
                this.art.fullscreen = !this.art.fullscreen;
                break;
        }
    }

    // Rewind 10 seconds
    rewind() {
        if (this.art) {
            const newTime = Math.max(0, this.art.currentTime - 10);
            this.art.seek = newTime;
            // Show feedback to user
            this.art.notice.show = 'Rewind 10s';
        }
    }

    // Forward 10 seconds
    forward() {
        if (this.art) {
            const newTime = Math.min(this.art.duration, this.art.currentTime + 10);
            this.art.seek = newTime;
            // Show feedback to user
            this.art.notice.show = 'Forward 10s';
        }
    }

    // Save playback position
    savePlaybackPosition() {
        if (!this.art || !playerConfig.playbackMemory.enabled) return;

        const progress = {
            url: this.art.url,
            currentTime: this.art.currentTime,
            duration: this.art.duration,
            timestamp: Date.now()
        };

        try {
            const key = `${playerConfig.playbackMemory.storageKey}-${btoa(this.art.url)}`;
            localStorage.setItem(key, JSON.stringify(progress));
        } catch (error) {
            console.error('Failed to save playback position:', error);
        }
    }

    // Restore playback position
    restorePlaybackPosition(videoUrl) {
        if (!playerConfig.playbackMemory.enabled) return;

        try {
            const key = `${playerConfig.playbackMemory.storageKey}-${btoa(videoUrl)}`;
            const saved = localStorage.getItem(key);
            if (saved) {
                const progress = JSON.parse(saved);
                // Only restore if the video was watched in the last 7 days
                if (Date.now() - progress.timestamp < 7 * 24 * 60 * 60 * 1000) {
                    this.art.once('ready', () => {
                        this.art.seek = progress.currentTime;
                    });
                }
            }
        } catch (error) {
            console.error('Failed to restore playback position:', error);
        }
    }

    // Handle HLS errors
    handleHLSError(event, data) {
        if (data.fatal) {
            switch (data.type) {
                case Hls.ErrorTypes.NETWORK_ERROR:
                    this.showError('Network error, trying to recover...');
                    this.hls.startLoad();
                    break;
                case Hls.ErrorTypes.MEDIA_ERROR:
                    this.showError('Media error, trying to recover...');
                    this.hls.recoverMediaError();
                    break;
                default:
                    this.showError('An error occurred while playing the video');
                    break;
            }
        }
    }

    // Validate URL
    isValidUrl(url) {
        try {
            new URL(url);
            return true;
        } catch {
            return false;
        }
    }

    // Show loading spinner
    showLoading() {
        if (this.loadingSpinner) {
            this.loadingSpinner.style.display = 'block';
        }
    }

    // Hide loading spinner
    hideLoading() {
        if (this.loadingSpinner) {
            this.loadingSpinner.style.display = 'none';
        }
    }

    // Show error message
    showError(message) {
        if (this.errorMessage) {
            this.errorMessage.textContent = message;
            this.errorMessage.style.display = 'block';
            this.hideLoading();
        }
    }

    // Destroy player instance
    destroy() {
        if (playerConfig.playbackMemory.enabled) {
            this.savePlaybackPosition();
        }
        if (this.hls) {
            this.hls.destroy();
        }
        if (this.art) {
            this.art.destroy();
        }
    }
} 